﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;

namespace MetroLightsOut
{
    // It really makes sense to inherit from level...
    // This would be a clean refactor, leaving separate for now
    class LevelSolver
    {
        Level _theLevel;
        int[][] _initialState;
        int rows, cols;
        HashSet<long> tested;
        
        // the maximum graph depth traversal
        // There's a sweet spot:
        // In games with more moves to complete, this will find an unoptimized solution faster
        // In games with fewer moves to complete, this will cause false starts..
        static int maxdepth = 15;
        int solutionsFound;        

        struct Move{            
            public int row;
            public int col;
        };        
        
        // LevelSolver solves levels...
        // 
        public LevelSolver(Level theLevel)
        {
            // this copies the level, right?
            _theLevel = theLevel;
            rows = theLevel.getRows();
            cols = theLevel.getCols();
            _initialState = theLevel.getGameBoard();

            solutionsFound = 0;

            // optimization to prevent circular traversal
            tested = new HashSet<long>();
        }

        // Calling solve will trigger a graph traversal using the helper
        // function
        public void Solve()
        {
            // Let's see if our work is already done :)
            if(!_theLevel.isComplete()){
                SolverState theState = new SolverState(_initialState, rows, cols);
                SolveHelper(theState);
            }
        }

        // TODO: Depth first vs breadth first...
        void SolveHelper(SolverState theState)
        {
            if (solutionsFound > 0) { return; }
            // level complete?
            // the stack of moves has the solution            
            if (theState.isComplete())
            {
                // yay!
                solutionsFound = solutionsFound+1;
                theState.ListMoveHistory();
                return;
            }

            // hash the state / test for circular states
            long stateHash = theState.stateToHash();
            if (tested.Contains(stateHash) || (theState.getMoveCount() > maxdepth - 1))
            {
                theState = null;
                return;
            }
            else
            {
                // Mark!
                tested.Add(theState.stateToHash());
            }

            // Step 1:
            // Get the available moves
            List<Move> available = theState.findPotentialMoves();            

            // Step 2:            
            // Simulate all promising moves, store in list
            List<SolverState> simulatedMoves = new List<SolverState>(); 
            foreach (Move m in available)
            {
                if (solutionsFound > 0) { 
                    return; 
                }
                SolverState toTest = new SolverState(theState);
                toTest.simulateMove(m);

                if (m.row == m.col && m.row == 0)
                {
                    // Freak out, you have entered the weird state
                }

                if (toTest.isComplete())
                {
                    // yay!
                    solutionsFound = solutionsFound+1;
                    toTest.ListMoveHistory();
                    return;
                }
                else
                {
                    // free
                    simulatedMoves.Add(toTest);
                }            
            }

            
            simulatedMoves.Sort(new SolverStateComparer());

            // TODO: can we free theState?
            foreach (SolverState toTest in simulatedMoves)
            {
                if (solutionsFound > 0) { return; }
                                                
                SolveHelper(toTest);
            }           
        }

        class SolverState
        {                        
            private Move[] currentMoveList;
            private int[][] _state;            
            int rows, cols, moveCount;


            // TODO: what happens if we share state? 
            // What happens if we don't manually copy state?
            public SolverState(SolverState s)
            {
                moveCount = s.moveCount;
                rows = s.rows;
                cols = s.cols;
                currentMoveList = new Move[maxdepth];
                _state = new int[rows][];                

                for (int i = 0; i < moveCount; i++)
                {
                    currentMoveList[i] = s.currentMoveList[i];
                }

                for (int row = 0; row < rows; row++)
                {
                    _state[row] = new int[cols];
                    for (int col = 0; col < cols; col++)
                    {
                        _state[row][col] = s._state[row][col];
                    }
                }
            }

            // Technically this is a constructor that could be used by the
            // game board or the level.
            // TODO: Potentially refactor with OO-design based on 
            // "LevelSolver inherits from Level"
            public SolverState(int[][] state, int _rows, int _cols)
            {                
                currentMoveList = new Move[maxdepth];
                moveCount = 0;
                rows = _rows;
                cols = _cols;

                _state = new int[rows][];

                for (int row = 0; row < rows; row++)
                {
                    _state[row] = new int[cols];
                    for (int col = 0; col < cols; col++)
                    {
                        _state[row][col] = state[row][col];
                    }
                }
            }


            // Optimization (1)
            // Let's try and avoid testing empty squares...
            // This will only return the moves that will remove a light.             
            //   
            public List<Move> findPotentialMoves()
            {
                List<Move> moves = new List<Move>();

                for (int row = 0; row < rows; row++)
                {
                    for (int col = 0; col < cols; col++)
                    {
                        // Let's only pick moves that will turn off a light...

                        // Center
                        bool potentialMove = false;
                        if (_state[row][col] == 1)
                        {
                            potentialMove = true;
                        }

                        // TOP 
                        if ((row-1) > 0)
                        {
                            if (_state[row - 1][col] == 1)
                            {
                                potentialMove = true;
                            }
                        }
                        // Bottom
                        if ((row+1) < rows)
                        {
                            if (_state[row + 1][col] == 1)
                            {
                                potentialMove = true;
                            }
                        }
                        // LEFT
                        if ((col-1) > 0)
                        {
                            if (_state[row][col - 1] == 1)
                            {
                                potentialMove = true;
                            }
                        }
                        // RIGHT
                        if ((col+1) < cols)
                        {
                            if (_state[row][col + 1] == 1)
                            {
                                potentialMove = true;
                            }
                        }
                        if (potentialMove)
                        {
                            Move m = new Move();
                            m.row = row;
                            m.col = col;
                            moves.Add(m);
                        }
                    }
                }
                return moves;
            }

            // Just simulates a move in the state
            // Note: This does change the state, so perhaps this should be named
            //       performMove indicating that it's not something like a "peek"
            //       Turning this into a peek would allow us to avoid creating
            //       two simulator allocations during graph traversal...
            public void simulateMove(Move m)
            {
                currentMoveList[moveCount] = m;
                moveCount = moveCount + 1;
                
                toggleSpace(m.row - 1, m.col); // top
                toggleSpace(m.row, m.col - 1); // left
                toggleSpace(m.row, m.col);     // center
                toggleSpace(m.row, m.col + 1); // right
                toggleSpace(m.row + 1, m.col); // bottom
            }


            // Helper function for simulate move
            // Performs bounds checking so the caller doesn't have to :)
            //
            void toggleSpace(int row,int col){
                if (row >= 0 && row < rows && col >= 0 && col < cols)
                {
                    if (_state[row][col] == 0)
                    {
                        _state[row][col] = 1;
                    }
                    else
                    {
                        _state[row][col] = 0;
                    }
                }
            }


            // Lists all the moves into debug
            // Todo: Play back the moves using the game...
            public void ListMoveHistory()
            {
                if (currentMoveList != null)
                {
                    Debug.WriteLine("------- Solution found --------");
                    for (int i=0; i < moveCount; i++)
                    {
                        Move m = currentMoveList[i];
                        Debug.WriteLine("Row: " + m.row + " Col: " + m.col);
                    }
                }
            }

            // Determine whether the game is complete
            // Alternative solution could be to use stateToHash() == 0
            //
            public bool isComplete()
            {
                for (int row = 0; row < rows; row++)
                {
                    for (int col = 0; col < cols; col++)
                    {
                        if (_state[row][col] != 0)
                        {
                            return false;
                        }
                    }
                }
                return true;
            }

            // Optimization (0) avoid circular logic
            // 
            // Turn the board into a state represented by a long
            // note this is limited to 64 bits (squares)
            // TODO: cache the state on first run, slight perf improvement
            public long stateToHash()
            {
                long hashValue = 0;

                int count = 0;

                for (int row = 0; row < rows; row++)
                {
                    for (int col = 0; col < cols; col++)
                    {
                        hashValue += (long)((_state[row][col] == 0) ? 0 : Math.Pow((double)2, (double)count));
                        count++;
                    }
                }

                return hashValue;
            }
            // Optimization (2) try and make the "smartest" move
            //
            // For now, just sum everything up and return the count of ones lower values preferred over higher
            // 
            // couple ideas:
            //  1) Perform two depth searches and sum the two
            //  2) Use different values for "adjacent" lights (those that could be turned off in 1 move)
            public int stateToRanking(){
                int sum = 0;
                for (int row = 0; row < rows; row++){
                    for (int col = 0; col < cols; col++)
                    {
                        sum += _state[row][col];
                    }
                }

                return sum;
            }


            
            public int getMoveCount()
            {
                return moveCount;
            }
        }

        public class SolverStateComparer : IComparer<SolverState>
        {
            int IComparer<SolverState>.Compare(SolverState left, SolverState right)
            {
                return left.stateToRanking() - right.stateToRanking();
            }
        }
    }
}
